package com.reconciliation.recfj.service;

import com.reconciliation.recfj.entity.RecExcel;
import com.reconciliation.recfj.entity.RecXml;
import com.reconciliation.recfj.entity.ReturnValue;
import com.reconciliation.recfj.enums.RecItem;
import com.reconciliation.recfj.enums.RecTable;
import com.reconciliation.recfj.mapper.RecMapper;
import com.reconciliation.recfj.util.DateUtils;
import com.reconciliation.recfj.util.ExcelUtils;
import com.reconciliation.recfj.util.XmlBuilderUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.ibatis.session.ExecutorType;
import org.apache.ibatis.session.SqlSession;
import org.apache.ibatis.session.SqlSessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;

/**
 * 对账实现类
 *
 * @author 史连宁
 */
@Service
public class RecService {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Resource
    private SqlSessionFactory sqlSessionFactory;
    @Resource
    private SendWebService sendWebService;
    /*@Value("${excel.directory.path}")
    private String excelPath;*/

    public ReturnValue createRecDataToExcel(RecExcel rec) {
        ReturnValue rtv = ReturnValue.fail();

        RecTable[] recTableList = RecTable.values();
        RecItem[] recItemList = RecItem.values();
        if (rec != null && !ObjectUtils.isEmpty(rec.getThirdTable())) {
            recTableList = new RecTable[]{RecTable.parse(rec.getThirdTable())};
        }
        if (rec != null && !ObjectUtils.isEmpty(rec.getRecItem())) {
            recItemList = new RecItem[]{RecItem.parse(rec.getRecItem())};
        }
        List<RecExcel> list = new ArrayList<>();
        SqlSession session = sqlSessionFactory.openSession(ExecutorType.BATCH, false);
        RecMapper recMapper = session.getMapper(RecMapper.class);
        try {
            for (RecTable recTable : recTableList) {
                for (RecItem recItem : recItemList) {
                    if (recItem.getFunc().equals("getRecAll")) {
                        //sql类型
                        if (recTable.getSqlType().equals("1")) {
                            List<RecExcel> recAll1 = recMapper.getRecAll1(recTable.getSysTable());
                            list.addAll(handleRecExcelData(recAll1, recItem, recTable));
                        }
                        if (recTable.getSqlType().equals("2")) {
                            List<RecExcel> recAll2 = recMapper.getRecAll2(recTable.getSysTable());
                            list.addAll(handleRecExcelData(recAll2, recItem, recTable));
                        }
                    } else if (recItem.getFunc().equals("getRecDay")) {
                        //sql类型
                        if (recTable.getSqlType().equals("1")) {
                            List<RecExcel> recDay1 = recMapper.getRecDay1(recTable.getSysTable());
                            list.addAll(handleRecExcelData(recDay1, recItem, recTable));
                        }
                        if (recTable.getSqlType().equals("2")) {
                            List<RecExcel> recDay2 = recMapper.getRecDay2(recTable.getSysTable());
                            list.addAll(handleRecExcelData(recDay2, recItem, recTable));
                        }
                    } else if (recItem.getFunc().equals("getRecMon")) {
                        //sql类型
                        if (recTable.getSqlType().equals("1")) {
                            List<RecExcel> recMon1 = recMapper.getRecMon1(recTable.getSysTable());
                            list.addAll(handleRecExcelData(recMon1, recItem, recTable));
                        }
                        if (recTable.getSqlType().equals("2")) {
                            List<RecExcel> recMon2 = recMapper.getRecMon2(recTable.getSysTable());
                            list.addAll(handleRecExcelData(recMon2, recItem, recTable));
                        }
                    }
                }
            }
            session.commit();
        } catch (Exception e) {
            logger.error("汇聚平台对账数据生成失败", e);
            rtv.setMsg(e.getMessage());
            session.rollback();
        } finally {
            if (session != null) {
                session.close();
            }
        }
        if (CollectionUtils.isEmpty(list)) {
            return ReturnValue.ok("无数据");
        }
        //将数据写到excel文件: RecData_20190701140506.xlsx
        writeDataToExcel(list);
        //由于推送数据到汇聚平台限制每次推送最多500条，将list分400条子list推送到汇聚平台
        List<List<RecExcel>> partitionList = ListUtils.partition(list, 400);
        //将数据封装成XML格式
        String type = "add";
        partitionList.forEach(subList -> {
            List<RecXml> recXmlList = packRecXmlData(subList, type);
            String xmlStr = XmlBuilderUtils.buildXml(recXmlList);
            // 发送到汇聚平台
            boolean flag = sendWebService.pushXmlData(xmlStr);
            if (flag) {
                rtv.setSuccess(true);
                rtv.setMsg("发送到汇聚平台成功！");
            } else {
                rtv.setSuccess(false);
                rtv.setMsg("发送到汇聚平台失败！");
            }
        });
        return rtv;
    }


    /**
     * 将数据封装成XML格式
     *
     * @param list
     * @param type
     * @return
     */
    private List<RecXml> packRecXmlData(List<RecExcel> list, String type) {
        List<RecXml> recXmlList = new ArrayList<>();
        if (ObjectUtils.isEmpty(list)) {
            return null;
        }
        list.forEach(recExcel -> {
            RecXml recXml = new RecXml();
            recXml.setType(type);
            recXml.setPrimaryKey(recExcel.getId());
            recXml.setIdxNo(recExcel.getRecId());
            recXml.setIdxType(recExcel.getRecItem());
            recXml.setTableName(recExcel.getThirdTable());
            recXml.setDateTime(recExcel.getDataTimeStr());
            recXml.setDataArea(recExcel.getDataArea());
            recXml.setOtherDim(recExcel.getOtherDimension());
            recXml.setDataCount(String.valueOf(recExcel.getValue()));
            Date execDate = recExcel.getExecDate();
            String execDateStr = DateUtils.format(execDate, DateUtils.P_YYYY_MM_DD_HH_MM_SS);
            recXml.setExecTime(execDateStr);
            recXml.setMaxUpdateTime(DateUtils.format(recExcel.getLatestUpdateDate(), DateUtils.P_YYYY_MM_DD_HH_MM_SS));
            recXml.setRemark(recExcel.getRemark());
            recXmlList.add(recXml);
        });
        return recXmlList;
    }


    /**
     * 将数据写到excel  文件格式: RecData_20190701140506.xlsx
     *
     * @param list
     */
    private void writeDataToExcel(List<RecExcel> list) {
        //创建目录
        String excelPath = System.getProperty("user.dir") + "/exportExcel";
        File dir = new File(excelPath);
        if (!dir.exists()) {
            dir.mkdir();
        }
        String fileName = "RecData" + "_" + DateUtils.ConvertDateToYYYYMMddHHmmss(new Date());
        //创建文件
        File file = new File(dir, fileName + ".xlsx");
        ExcelUtils.writeExcel(file, list);
    }

    /**
     * 封装数据
     *
     * @param recData
     * @param recItem
     * @param recTable
     */
    private List<RecExcel> handleRecExcelData(List<RecExcel> recData, RecItem recItem, RecTable recTable) {
        Date now = new Date();
        if (!CollectionUtils.isEmpty(recData)) {
            recData = recData.stream().map(recAll -> {
                RecExcel recExcel = new RecExcel();
                recExcel.setId(createRandomId());
                recExcel.setRecId(recItem.getId() + "_" + recTable.getHjTable());
                recExcel.setRecItem(recItem.getName());
                recExcel.setThirdTable(recTable.getHjTable());
                //指标“IDX1数据总量”此列不用填
                if (!recItem.getId().equals("IDX1")) {
                    recExcel.setDataTimeStr(createDataTimeStr(recAll.getTbrq()));
                }
                recExcel.setExecDate(now);
                recExcel.setDataArea(recTable.getSysTable());
                recExcel.setOtherDimension(recTable.getTableName());
                recExcel.setValue(recAll.getValue());
                return recExcel;
            }).collect(Collectors.toList());
        }else{
            recData = new ArrayList<>();
            RecExcel recExcel = new RecExcel();
            recExcel.setId(createRandomId());
            recExcel.setRecId(recItem.getId() + "_" + recTable.getHjTable());
            recExcel.setRecItem(recItem.getName());
            recExcel.setThirdTable(recTable.getHjTable());
            //指标“IDX1数据总量”此列不用填
            if (!recItem.getId().equals("IDX1")) {
                recExcel.setDataTimeStr(createDataTimeStr(DateUtils.format(now, DateUtils.P_YYYY_MM_DD)));
            }
            recExcel.setExecDate(now);
            recExcel.setDataArea(recTable.getSysTable());
            recExcel.setOtherDimension(recTable.getTableName());
            recExcel.setValue(0L);
            recData.add(recExcel);
        }
        return recData;
    }

    /**
     * 生成更新时间：格式为yyyyMM/yyyyMMdd
     *
     * @param tbrq
     * @return
     */
    private String createDataTimeStr(String tbrq) {
        if (ObjectUtils.isEmpty(tbrq)) {
            return "";
        }
        if (tbrq.length() == 7) {
            return DateUtils.convertToYYYYMM(tbrq);
        }
        if (tbrq.length() == 10) {
            return DateUtils.convertToYYYYMMdd(tbrq);
        }
        return null;
    }

    /**
     * 生成随机ID
     *
     * @return
     */
    private String createRandomId() {
        // 获取当前的Instant时间
        String time = DateUtils.ConvertDateToYYYYMMddHHmmss(new Date());
        // 生成六位随机数
        Random random = new Random();
        int randomNumber = 100000 + random.nextInt(900000); // 生成范围在100000到999999之间的随机数
        // 拼接结果
        String randomId = time + String.format("%06d", randomNumber); // 确保格式为六位
        return randomId;
    }

    public static void main(String[] args) {
        String time = DateUtils.ConvertDateToYYYYMMddHHmmss(new Date());
        System.out.printf(time);
    }

    public ReturnValue testAddXmlToWebService() {
        SqlSession session = sqlSessionFactory.openSession(ExecutorType.BATCH, false);
        RecMapper recMapper = session.getMapper(RecMapper.class);
        try {
            List<RecExcel> recAll1 = recMapper.getRecAll1("SD_MS_DFS");
            logger.info("查询测试结果:" + recAll1);
            session.commit();
        }catch (Exception e) {

            session.rollback();
        } finally {
            if (session != null) {
                session.close();
            }
        }



        ReturnValue rvt = new ReturnValue();
        List<RecExcel> list = createTestRecExcelData();
        String type = "add";
        List<RecXml> recXmlList = packRecXmlData(list, type);
        String xmlStr = XmlBuilderUtils.buildXml(recXmlList);
        // 发送到汇聚平台
        boolean flag = sendWebService.pushXmlData(xmlStr);
        System.out.printf("xml文件是：" + xmlStr);
        if (flag) {
            rvt.setMsg("推送add测试数据成功");
            rvt.setSuccess(true);
        } else {
            rvt.setMsg("推送add测试数据失败");
            rvt.setSuccess(false);
        }
        return rvt;
    }

    /**
     * 生成测试数据
     *
     * @return
     */
    private List<RecExcel> createTestRecExcelData() {

        List<RecExcel> list = new ArrayList<>();
        RecExcel recExcel = new RecExcel();
        recExcel.setId("20241115142624296948");
        recExcel.setRecId("IDX2_table34904");
        recExcel.setThirdTable("table34904");
        recExcel.setExecDate(new Date());
        recExcel.setRecItem("IDX2创建时间月统计值");
        recExcel.setDataTimeStr("");
        recExcel.setValue(100L);
        recExcel.setDataArea("");
        recExcel.setOtherDimension("");
        recExcel.setLatestUpdateDate(new Date());
        list.add(recExcel);
        return list;
    }

    public ReturnValue testDelXmlToWebService() {
        ReturnValue rvt = new ReturnValue();
        List<RecExcel> list = createTestRecExcelData();
        String type = "delete";
        List<RecXml> recXmlList = packRecXmlData(list, type);
        String xmlStr = XmlBuilderUtils.buildXml(recXmlList);
        // 发送到汇聚平台
        boolean flag = sendWebService.pushXmlData(xmlStr);
        System.out.printf("xml文件是：" + xmlStr);
        if (flag) {
            rvt.setMsg("推送delete测试数据成功!");
            rvt.setSuccess(true);
        } else {
            rvt.setMsg("推送delete测试数据失败!");
            rvt.setSuccess(false);
        }
        return rvt;
    }
}
