/**
 * 日期工具
 */
package com.reconciliation.recfj.util;


import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 日期处理工具类
 */
public class DateUtils {

    public static final String P_YYYY = "yyyy";

    public static final String P_MM = "MM";

    public static final String P_YYYYMM = "yyyyMM";

    public static final String P_MM_DD_HH_MM = "MM/dd HH:mm";

    public static final String P_HH_MM = "HH:mm";

    public static final String P_YYYYMMDD = "yyyyMMdd";

    public static final String P_YYYYMMDDHH = "yyyyMMddHH";

    public static final String P_YYYY_MM_DD = "yyyy-MM-dd";

    public static final String P_YYYY_MM_DD_1 = "yyyy/MM/dd";

    public static final String P_YYYY_MM_DD_2 = "yy-MM-dd";

    public static final String P_YYYY_MM = "yyyy-MM";


    public static final String P_YY_MM = "yyMM";

    public static final String P_MM_DD = "MM-dd";
    public static final String P_MM_DD_x = "MM/dd";

    public static final String P_MM_DD_CN = "MM月dd日";

    public static final String P_YYYY_MM_DD_CN = "yyyy年MM月dd日";

    public static final String P_YYYY_MM_CN = "yyyy年MM月";

    public static final String P_YYYY_MM_DD_E_HH_MM_CN = "yyyy年MM月dd日 E HH:mm";

    public static final String P_YYYY_MM_DD_HH_MM = "yyyy-MM-dd HH:mm";

    public static final String P_YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    public static final String P_YYYY_MM_DD_HH_MM_CN = "yyyy年MM月dd日 HH时mm分";

    public static final String P_YYYY_MM_DD_HH_MM_SS_CN = "yyyy年MM月dd日 HH时mm分ss秒";

    public static final String P_YYYY_MM_DD_HH_MM_SS_NO_DELIMITER = "yyyyMMddHHmmss";

    public static final String P_YYYY_MM_DD_HH_MM_NO_DELIMITER = "yyyyMMddHHmm";

    public static final String P_HH_MM_SS = "HH:mm:ss";

    public static final String DAY_BEGIN_TIME = " 00:00:00";

    public static final String DAY_END_TIME = " 23:59:59";

    private static final String FORMAT_NULL_DATE = "--";

    public static final String P_YY = "yy";

    public static final String MMM_d_yyyy_K_m_s_a = "MMM d, yyyy K:m:s a";

    /**
     * 将日期格式化为字符串
     *
     * @param date
     * @param pattern
     * @return
     */
    public static String format(Date date, String pattern) {
        String s = "";
        if (date != null) {
            return format(date, pattern, s);
        }
        return s;
    }

    /**
     * 将日期格式化为字符串:格式化为yyyyMMddHHmmss
     *
     * @param date
     * @return
     */
    public static String ConvertDateToYYYYMMddHHmmss(Date date) {
        return format(date, P_YYYY_MM_DD_HH_MM_SS_NO_DELIMITER);
    }

    /**
     * 将日期格式化为字符串
     *
     * @param date
     * @return
     */
    public static String format(Date date) {
        return format(date, P_YYYY_MM_DD);
    }

    /**
     * 将字符串转换为日期
     *
     * @param date
     * @param pattern
     * @return
     */
    public static Date parse(Date date, String pattern) {
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        Date newDate = null;
        try {
            newDate = sdf.parse(sdf.format(date));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return newDate;
    }

    /**
     * 将日期格式化为字符串
     *
     * @param date
     * @param pattern
     * @return
     */
    public static String format(Date date, String pattern, String errorFormat) {
        String s = errorFormat;
        if (date != null) {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            try {
                s = sdf.format(date);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return s;
    }

    /**
     * 将字符串转换为日期
     *
     * @param pattern
     * @return
     */
    public static Date parse(String dateStr, String pattern) {
        Date d = null;
        if (dateStr != null && dateStr.length() > 0) {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            try {
                d = sdf.parse(dateStr);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return d;
    }

    /**
     * 校验日期是否是yyyy-MM-dd格式
     *
     * @param dateStr
     * @return
     */
    public static boolean checkFormat(String dateStr) {
        String timeRegex1 = "(([0-9]{3}[1-9]|[0-9]{2}[1-9][0-9]{1}|[0-9]{1}[1-9][0-9]{2}|[1-9][0-9]{3})-(((0[13578]|1[02])-(0[1-9]|[12][0-9]|3[01]))|" +
                "((0[469]|11)-(0[1-9]|[12][0-9]|30))|(02-(0[1-9]|[1][0-9]|2[0-8]))))|((([0-9]{2})(0[48]|[2468][048]|[13579][26])|" +
                "((0[48]|[2468][048]|[3579][26])00))-02-29)$";
        return Pattern.matches(timeRegex1, dateStr);
    }

    /**
     * 时间格式转字符串
     *
     * @return
     */
    public static Date parseTime(String in) {
        Date d = null;
        if (in != null && in.length() > 0) {

            String date = "";

            if (in.contains("12:00:00 AM")) date = in.replaceAll("12:00:00 AM", "00:00:00 AM");
            else if (in.contains("12:00:00 PM")) date = in.replaceAll("PM", "AM");
            else date = in;

            SimpleDateFormat sdf = new SimpleDateFormat(MMM_d_yyyy_K_m_s_a, Locale.ENGLISH);
            try {
                d = sdf.parse(date);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return d;
    }

    /**
     * 获取当前时间
     *
     * @return
     */
    public static Date getCurrentDateTime() {
        Calendar calendar = Calendar.getInstance();
        return calendar.getTime();
    }

    /**
     * 获取当前日期
     *
     * @return
     */
    public static Date getCurrentDate() {
        Date now = getCurrentDateTime();
        return strToDate(dateToStr(now));
    }

    /**
     * 获取当前日期
     *
     * @return
     */
    public static String getCurrentDateStr() {
        Date now = getCurrentDateTime();
        return dateToStr(now);
    }

    /**
     * 获取当前时间
     *
     * @return
     */
    public static String getCurrentDateTimeStr() {
        return format(getCurrentDateTime(), P_YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 获取当前日期
     *
     * @return
     */
    public static String getCurrentDateCN() {
        Date now = getCurrentDateTime();
        return format(now, P_YYYY_MM_DD_CN);
    }

    /**
     * 字符串转日期，默认格式
     *
     * @param s
     * @return
     */
    public static Date strToDate(String s) {
        Date d = null;

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            d = sdf.parse(s);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return d;
    }

    /**
     * 字符串转时间，默认格式
     *
     * @param s
     * @return
     */
    public static Date strToTime(String s) {
        Date d = null;

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        try {
            d = sdf.parse(s);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return d;
    }

    /**
     * 字符串转日期，带时分秒
     *
     * @param s
     * @return
     */
    public static Date strToDateHMS(String s) {
        String ss = s;
        if (s.length() == 16) {
            ss += ":00";
        }
        Date d = null;
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        try {
            d = sdf.parse(ss);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return d;
    }

    /**
     * 日期转字符串，默认格式
     *
     * @param date
     * @return
     */
    public static String dateToStr(Date date) {
        String s = "";
        if (date == null) {
            return "";
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            s = sdf.format(date);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return s;
    }

    /**
     * 获取当前周的周一
     *
     * @return
     */
    public static Date getThisWeekMonday() {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        return strToDate(dateToStr(calendar.getTime()));
    }

    /**
     * 获取当前周的周日
     *
     * @return
     */
    public static Date getThisWeekSunday() {
        Calendar calendar = Calendar.getInstance();
        int day_of_week = calendar.get(Calendar.DAY_OF_WEEK) - 1;
        if (day_of_week == 0)
            day_of_week = 7;
        calendar.add(Calendar.DATE, -day_of_week + 7);
        return strToDate(dateToStr(calendar.getTime()));
    }

    /**
     * 判断某个日期是否在开始和结束日期范围内，精确到天
     *
     * @param now
     * @param startDate
     * @param endDate
     * @return
     */
    public static boolean isBetweenTwoDate(Date now, Date startDate, Date endDate) {
        if (startDate != null) {
            startDate = strToDate(dateToStr(startDate) + DAY_BEGIN_TIME);
        }
        if (endDate != null) {
            endDate = strToDateHMS(dateToStr(endDate) + DAY_END_TIME);
        }
        return isBetweenTwoDateHMS(now, startDate, endDate);
    }

    /**
     * 判断某个时间是否在开始和结束时间范围内，精确到毫秒
     *
     * @param now
     * @param startDate
     * @param endDate
     * @return
     */
    public static boolean isBetweenTwoDateHMS(Date now, Date startDate, Date endDate) {
        if (startDate == null && endDate == null) {
            return true;
        } else if (startDate != null && endDate != null) {
            return now.getTime() >= startDate.getTime() && now.getTime() <= endDate.getTime();
        } else if (startDate != null) {
            return now.getTime() >= startDate.getTime();
        } else if (endDate != null) {
            return now.getTime() <= endDate.getTime();
        }

        return false;
    }

    /**
     * 根据年数，取得年份列表。
     *
     * @param yearNum 要取得的年数。
     * @author YangZhenghua
     * @date 2014-7-15
     */
    public static List<String> getLastYearsByNum(int yearNum) {
        int currentYear;
        List<String> lstTenYears = new ArrayList<String>();
        Calendar cal = Calendar.getInstance();
        currentYear = cal.get(Calendar.YEAR);

        for (int i = 0; i < yearNum; i++) {
            lstTenYears.add((currentYear - i) + "");
        }

        return lstTenYears;
    }

    /**
     * 取得最近十年的年份。
     *
     * @author YangZhenghua
     * @date 2014-7-15
     */
    public static List<String> getLastTenYears() {
        return getLastYearsByNum(10);
    }

    /**
     * 判断是否是闰年
     *
     * @param year
     * @return
     * @author YangZhenghua
     * @date 2014-7-21
     */
    public static boolean isLeapYear(int year) {
        if (year % 4 == 0 && year % 100 != 0 || year % 400 == 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 计算给定的年，某个月 有多少天。
     *
     * @param year
     * @param month
     * @return
     * @author YangZhenghua
     * @date 2014-7-21
     */
    public static int calDaysInYearAndMonth(int year, int month) {

        if (month == 2 && isLeapYear(year)) {
            return 29;
        } else if (month == 2 && !isLeapYear(year)) {
            return 28;
        } else if (month == 4 || month == 6 || month == 9 || month == 11) {
            return 30;
        } else {
            return 31;
        }
    }

    /**
     * 取出指定时间的月和日。
     *
     * @param dateTime
     * @return
     * @author YangZhenghua
     * @date 2014-7-28
     */
    public static String getMonthAndDay(java.sql.Date dateTime) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(dateTime);

        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DATE);

        return month + "-" + day;
    }

    /**
     * 取出指定时间的天。
     *
     * @param dateTime
     * @return
     * @author YangZhenghua
     * @date 2014-7-29
     */
    public static int getDay(java.sql.Date dateTime) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(dateTime);
        int day = calendar.get(Calendar.DATE);

        return day;
    }

    /**
     * 取出指定时间在一年中的天数。
     *
     * @param dateTime
     * @return
     * @author YangZhenghua
     * @date 2014-7-29
     */
    public static int getDayOfYear(Date dateTime) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(dateTime);
        int day = calendar.get(Calendar.DAY_OF_YEAR);

        return day;
    }

    /**
     * 比较两个日期的月和日是否相同。
     *
     * @param dateFromTime
     * @param dateToTime
     * @return
     * @author YangZhenghua
     * @date 2014-7-28
     */
    public static boolean compareSameMonthAndDay(java.sql.Date dateFromTime, java.sql.Date dateToTime) {
        return getMonthAndDay(dateFromTime).equals(getMonthAndDay(dateToTime));
    }

    /**
     * 日期加减法,得到ii天后的日期
     *
     * @param date 原日期
     * @param ii   向前或向后的天数
     * @return 得到ii天后的日期
     */
    public static Date addDay(Date date, int ii) {
        Calendar ins = Calendar.getInstance();
        ins.setTime(date);
        ins.set(Calendar.DAY_OF_YEAR, ins.get(Calendar.DAY_OF_YEAR) + ii);
        return ins.getTime();
    }

    /**
     * 日期加减法,得到ii月后的日期
     *
     * @param date 原日期
     * @param ii   向前或向后的月数
     * @return 得到ii月后的日期
     */
    public static Date addMonth(Date date, int ii) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, ii);
        return calendar.getTime();
    }

    /**
     * 日期加减法,得到ii年后的日期
     *
     * @param date 原日期
     * @param ii   向前或向后的年数
     * @return 得到ii年后的日期
     */
    public static Date addYear(Date date, int ii) {
        if (null == date) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.YEAR, ii);
        return calendar.getTime();
    }

    /**
     * 分钟
     *
     * @param date
     * @param ii
     * @return
     */
    public static Date addMin(Date date, int ii) {
        Calendar ins = Calendar.getInstance();
        ins.setTime(date);
        ins.set(Calendar.MINUTE, ins.get(Calendar.MINUTE) + ii);
        return ins.getTime();
    }

    /**
     * 将格林时间字符串转换成yyyy-MM-dd日期格式字符串
     *
     * @param gmtDate DatePicker日期组件传来的格林时间字符串
     * @return strDate yyyy-MM-dd日期格式字符串
     * @throws ParseException
     */
    public static String gmtToStr(String gmtDate) throws ParseException {
        String strDate = "";
        SimpleDateFormat sdfGmt = new SimpleDateFormat("EEE MMM dd yyyy hh:mm:ss z", Locale.ENGLISH);
        SimpleDateFormat sdfDate = new SimpleDateFormat("yyyy-MM-dd");
        gmtDate = gmtDate.replace("GMT", "").replaceAll("\\(.*\\)", "");
        Date newDate = sdfGmt.parse(gmtDate);
        strDate = sdfDate.format(newDate);
        return strDate;
    }

    /**
     * 获取日期前后几天（前几天正数/后几天负数）
     *
     * @param num
     * @param pattern
     * @return
     */
    public static String nextLastDay(int num, String pattern) {
        System.currentTimeMillis();
        DateUtils.format(new Date(System.currentTimeMillis()), pattern);
        return DateUtils.format(new Date(System.currentTimeMillis() - new Long(num * 24L * 60 * 60 * 1000)), pattern);
    }

    public static void main(String[] args) {
//		System.out.print(DateUtils.format(DateUtils.parseTime("Jul 16, 2018 12:00:00 PM"),"yyyy-MM-dd HH:mm:ss"));
//		
//		System.out.println("Jul 8, 2018 1:00:00 PM".contains("12:00:00 AM"));
        //System.out.println(DateUtils.nextLastDay(1, P_YYYY_MM_DD));

        System.out.println(DateUtils.getMouthLastday());
    }

    /**
     * 判断一个时间是否在一个区间内
     *
     * @param nowTime
     * @param startTime
     * @param endTime
     * @return
     */
    public static boolean isEffectiveDate(Date nowTime, Date startTime, Date endTime) {
        if (nowTime.getTime() == startTime.getTime()
                || nowTime.getTime() == endTime.getTime()) {
            return true;
        }

        Calendar date = Calendar.getInstance();
        date.setTime(nowTime);

        Calendar begin = Calendar.getInstance();
        begin.setTime(startTime);

        Calendar end = Calendar.getInstance();
        end.setTime(endTime);

        if (date.after(begin) && date.before(end)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 根据时间获取当前是周几(数字)
     *
     * @param date
     * @return
     * @throws ParseException
     */
    public static int dayForWeek(String date) throws ParseException { // 必须yyyy-MM-dd
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar c = Calendar.getInstance();
        c.setTime(format.parse(date));
        int dayForWeek = 0;
        if (c.get(Calendar.DAY_OF_WEEK) == 1) {
            dayForWeek = 7;
        } else {
            dayForWeek = c.get(Calendar.DAY_OF_WEEK) - 1;
        }
        return dayForWeek;
    }

    /**
     * Aug 3, 2018 12:00:00 AM 转换为yyyy-MM-dd HH:mm:ss
     *
     * @param dateStr
     * @return
     * @throws ParseException
     */
    public static String dateSdfToStr(String dateStr) throws ParseException { // 必须yyyy-MM-dd
        SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        SimpleDateFormat sdf2 = new SimpleDateFormat("MMM d, yyyy K:m:s a", Locale.ENGLISH);
        Date date = sdf2.parse(dateStr);
        String sdfStr = sdf1.format(date);
        return sdfStr;
    }

    /**
     * 根据某一天获取前7天的日期
     *
     * @param dateStr
     * @return
     */
    public static String getMonDay(String dateStr) {
        Date date = DateUtils.strToDate(dateStr);
        Calendar calendarMon = Calendar.getInstance();
        calendarMon.setTime(date);
        calendarMon.add(Calendar.DATE, -7);
        String monDay = DateUtils.format(calendarMon.getTime(), DateUtils.P_YYYYMMDD);
        return monDay;
    }

    /**
     * 根据某一天获取后num天的日期
     *
     * @param dateStr
     * @return
     */
    public static String getNumDay(String dateStr, int num) {
        Date date = DateUtils.strToDate(dateStr);
        Calendar calendarMon = Calendar.getInstance();
        calendarMon.setTime(date);
        calendarMon.add(Calendar.DATE, num);
        String monDay = DateUtils.format(calendarMon.getTime(), DateUtils.P_YYYY_MM_DD);
        return monDay;
    }

    /**
     * 根据某一天获取前1天的日期
     *
     * @param dateStr
     * @return
     */
    public static String getSunDay(String dateStr) {
        Date date = DateUtils.strToDate(dateStr);
        Calendar calendarSun = Calendar.getInstance();
        calendarSun.setTime(date);
        calendarSun.add(Calendar.DATE, -1);
        String sunDay = DateUtils.format(calendarSun.getTime(), DateUtils.P_YYYYMMDD);
        return sunDay;
    }

    /**
     * 将Jan 28, 2014 11:24:18 AM转换为Tue Jan 28 11:24:18 CST 2014
     *
     * @param dateStr
     * @return
     */
    public static Date cstFormat(String dateStr) {
        SimpleDateFormat sdf = new SimpleDateFormat("MMM d, yyyy", Locale.ENGLISH);
        Date d2 = null;
        try {
            d2 = sdf.parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return d2;
    }

    /**
     * 转换 yyyy-MM-dd 到 yyyyMMdd
     *
     * @param dateStr
     * @return
     */
    public static String convertToYYYYMMdd(String dateStr) {
        SimpleDateFormat inputFormat = new SimpleDateFormat("yyyy-MM-dd");
        SimpleDateFormat outputFormat = new SimpleDateFormat("yyyyMMdd");

        // 解析输入格式并格式化成输出格式
        Date date = null;
        try {
            date = inputFormat.parse(dateStr);
        } catch (ParseException e) {
            return "";
        }
        return outputFormat.format(date);
    }

    /**
     * 转换 yyyy-MM 到 yyyyMM
     *
     * @param dateStr
     * @return
     */
    public static String convertToYYYYMM(String dateStr) {
        SimpleDateFormat inputFormat = new SimpleDateFormat("yyyy-MM");
        SimpleDateFormat outputFormat = new SimpleDateFormat("yyyyMM");

        // 解析输入格式并格式化成输出格式
        Date date = null;
        try {
            date = inputFormat.parse(dateStr);
        } catch (ParseException e) {
            return "";
        }
        return outputFormat.format(date);
    }

    /**
     * 将日期字符串转换成yyyyMMdd--日期格式字符串
     *
     * @param dateStr
     * @param fmt
     * @return
     */
    public static String dateToNum(String dateStr, String fmt) {
        Date date = DateUtils.strToDate(dateStr);
        String wu = DateUtils.format(date, fmt);
        return wu;
    }

    /**
     * 将格林时间字符串转换成yyyyMMdd日期格式字符串
     *
     * @param gmtDate DatePicker日期组件传来的格林时间字符串
     * @return strDate yyyyMMdd日期格式字符串
     * @throws ParseException
     */
    public static String gmtToNum(String gmtDate, String fmt) throws ParseException {
        String strDate = "";
        SimpleDateFormat sdfGmt = new SimpleDateFormat("EEE MMM dd yyyy hh:mm:ss z", Locale.ENGLISH);
        SimpleDateFormat sdfDate = new SimpleDateFormat(fmt);
        gmtDate = gmtDate.replace("GMT", "").replaceAll("\\(.*\\)", "");
        Date newDate = sdfGmt.parse(gmtDate);
        strDate = sdfDate.format(newDate);
        return strDate;
    }

    /**
     * 根据参数年月获取上一个月份
     * @param ym
     * @return
     * @throws Exception
     *//*
    public static String getLastMonthByYM(String ym) throws ParseException {
		SimpleDateFormat sdf = new SimpleDateFormat(DateUtils.P_YYYYMM);
		Calendar c = Calendar.getInstance();
		if(StringUtils.isNotBlank(ym)){
			Date date = sdf.parse(ym);
			c.setTime(date);
		}
		c.add(Calendar.MONTH, -1);
		return sdf.format(c.getTime());
    }
	*//**
     * 根据参数年月获取下一个月份
     * @param ym
     * @return
     * @throws Exception
     *//*
	public static String getNextMonthByYM(String ym) throws ParseException {
		SimpleDateFormat sdf = new SimpleDateFormat(DateUtils.P_YYYYMM);
		Calendar c = Calendar.getInstance();
		if(StringUtils.isNotBlank(ym)){
			Date date = sdf.parse(ym);
			c.setTime(date);
		}
		c.add(Calendar.MONTH, +1);
		return sdf.format(c.getTime());
	}*/

    /**
     * 根据参数年获取上一年
     * @param year
     * @return
     * @throws Exception
     *//*
	public static String getLastYearByYearStr(String year) throws ParseException {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy");
		Calendar c = Calendar.getInstance();
		if(StringUtils.isNotBlank(year)){
			Date date = sdf.parse(year);
			c.setTime(date);
		}
		c.add(Calendar.YEAR, -1);
		return sdf.format(c.getTime());
	}*/

    /**
     * get first date of given month and year
     *
     * @param year
     * @param month
     * @return
     */
    public String getFirstDayOfMonth(int year, int month) {
        String monthStr = month < 10 ? "0" + month : String.valueOf(month);
        return year + "-" + monthStr + "-" + "01";
    }

    /**
     * get the last date of given month and year
     *
     * @param year
     * @param month
     * @return
     */
    public String getLastDayOfMonth(int year, int month) {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, month - 1);
        calendar.set(Calendar.DATE, 1);
        calendar.add(Calendar.MONTH, 1);
        calendar.add(Calendar.DAY_OF_YEAR, -1);
        return calendar.get(Calendar.YEAR) + "-" + (calendar.get(Calendar.MONTH) + 1) + "-" +
                calendar.get(Calendar.DAY_OF_MONTH);
    }

    /**
     * get Calendar of given year
     *
     * @param year
     * @return
     */
    private static Calendar getCalendarFormYear(int year) {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        cal.set(Calendar.YEAR, year);
        return cal;
    }

    /**
     * get start date of given week no of a year
     *
     * @param year
     * @param weekNo
     * @return
     */
    public static String getStartDayOfWeekNo(int year, int weekNo) {
        Calendar cal = getCalendarFormYear(year);
        cal.set(Calendar.WEEK_OF_YEAR, weekNo);
        return cal.get(Calendar.YEAR) + "-" + (cal.get(Calendar.MONTH) + 1) + "-" +
                cal.get(Calendar.DAY_OF_MONTH);

    }

    /**
     * get the end day of given week no of a year.
     *
     * @param year
     * @param weekNo
     * @return
     */
    public static String getEndDayOfWeekNo(int year, int weekNo) {
        Calendar cal = getCalendarFormYear(year);
        cal.set(Calendar.WEEK_OF_YEAR, weekNo);
        cal.add(Calendar.DAY_OF_WEEK, 6);
        return cal.get(Calendar.YEAR) + "-" + (cal.get(Calendar.MONTH) + 1) + "-" +
                cal.get(Calendar.DAY_OF_MONTH);
    }

    /**
     * 根据最小值,最大值分段
     *
     * @param min
     * @param max
     * @param num
     * @return
     */
    public static List<String> price(long min, long max, int num) {

        ArrayList<String> arrayList = new ArrayList<>();

        //最大值除以分段数 得出 每段中增加值
        long addPrice = (max - min) / num;

        //一直增加值
        int totalPrice = (int) min;

        //记录上次的每段增加值
        int addPriceCopy = 0;

        for (int i = 0; i < num; i++) {
            addPriceCopy = totalPrice;
            totalPrice = (int) (totalPrice + addPrice);
            if (i == num - 1) {
                arrayList.add(1 + addPriceCopy + ";" + max);
            } else if (i != 0 && i != num - 1) {
                arrayList.add(1 + addPriceCopy + ";" + totalPrice);
            } else {
                arrayList.add(addPriceCopy + ";" + totalPrice);
            }
        }

        return arrayList;
    }

    /**
     * 将yyyyMMdd转换为yyyy-MM-dd
     *
     * @param num
     * @return
     */
    public static String numToStr(String num) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMdd");
        formatter.setLenient(false);
        Date newDate = new Date();
        try {
            newDate = formatter.parse(num);
            formatter = new SimpleDateFormat("yyyy-MM-dd");
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return formatter.format(newDate);
    }

    public static String getWeek(String dateStr) {

        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Date date = null;

        try {
            date = format.parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }

        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        int week = calendar.get(Calendar.WEEK_OF_YEAR) - 1;
        String weekStr = dateStr.substring(0, 4) + "-W" + week;
        return weekStr;
    }

    /**
     * 获取两个日期之间所有日期的集合
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static List<String> getBetweenDate(String startTime, String endTime) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        // 声明保存日期集合
        List<String> list = new ArrayList<String>();
        try {
            // 转化成日期类型
            Date startDate = sdf.parse(startTime);
            Date endDate = sdf.parse(endTime);

            //用Calendar 进行日期比较判断
            Calendar calendar = Calendar.getInstance();
            while (startDate.getTime() <= endDate.getTime()) {
                // 把日期添加到集合
                list.add(sdf.format(startDate));
                // 设置日期
                calendar.setTime(startDate);
                //把日期增加一天
                calendar.add(Calendar.DATE, 1);
                // 获取增加后的日期
                startDate = calendar.getTime();
            }
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 判断是否为标准时间类型
     *
     * @param str
     * @return
     */
    public static boolean isDate(String str) {
        boolean convertSuccess = true;
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        try {
            format.setLenient(false);
            format.parse(str);
        } catch (ParseException e) {
            convertSuccess = false;
        }
        return convertSuccess;
    }

    /**
     * 获取两个年份之间的全部年份
     *
     * @param startyear
     * @param endyear
     * @return
     */
    public static List<String> getyearList(String startyear, String endyear) {

        int start = Integer.parseInt(startyear);
        int end = Integer.parseInt(endyear);
        int year = end - start;
        List<String> yearList = new ArrayList<>();

        for (int i = 0; i <= year; i++) {
            int newyear = start + i;
            yearList.add(String.valueOf(newyear));
        }

        return yearList;
    }

    public static String getWeekAndYear(String date) {
        String result = "";
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(Calendar.MONDAY);
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");

        Date d = null;
        try {
            d = format.parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }

        cal.setTime(d);
        int week = cal.get(Calendar.WEEK_OF_YEAR);
        result = String.valueOf(week);

        if (result.length() < 2) {
            result = "0" + result;
        }

        return result;
    }

    /**
     * 根据报表开始时间,获取业务数据
     *
     * @param date
     * @return
     */
    public static String GetWeekOfYear(Date date) {
        Calendar c = new GregorianCalendar();
        c.setFirstDayOfWeek(Calendar.MONDAY);
        c.setMinimalDaysInFirstWeek(7);

        c.setTime(date);

        int weeknum = c.get(Calendar.WEEK_OF_YEAR);

        int vmonth = c.get(Calendar.MONTH) + 1;

        if (vmonth == 1 && weeknum > 6) {
            weeknum = 1;
        }
        String result = "";
        result = String.valueOf(weeknum);

//    if (result.length()<2) {
//    	result = "0" + result;
//		}
        return result;
    }

    /**
     * 业务时间获取下一期
     *
     * @param businesstime
     * @return
     */
    public static String GetNextBus(String businesstime) {

        String bus = businesstime.substring(6, businesstime.length() - 1);
        int ci = Integer.valueOf(bus) + 1;
        return businesstime.substring(0, 6) + ci + "期";
    }

    /**
     * 业务时间获取上一期
     *
     * @param businesstime
     * @return
     */
    public static String GetBeforeBus(String businesstime) {

        String bus = businesstime.substring(6, businesstime.length() - 1);
        int ci = Integer.valueOf(bus) - 1;
        return businesstime.substring(0, 6) + ci + "期";
    }

    /**
     * 获取两个业务时间中间数据
     *
     * @param b1
     * @param b2
     * @return
     */
    public static List<String> GetNextBusList(String b1, String b2) {

        String bus1 = b1.substring(6, b1.length() - 1);
        int ci1 = Integer.valueOf(bus1);
        String bus2 = b2.substring(6, b2.length() - 1);
        int ci2 = Integer.valueOf(bus2);
        int ci = ci2 - ci1;
        List<String> busList = new ArrayList<>();

        for (int i = 0; i <= ci; i++) {
            String businesstime = "";
            int c = ci1 + i;
            businesstime = b1.substring(0, 6) + c + "期";
            busList.add(businesstime);
        }
        return busList;
    }

    public static int getDay(String date) {

        Calendar ca = Calendar.getInstance();//创建一个日期实例

        ca.setTime(DateUtils.parse(date, DateUtils.P_YYYY_MM_DD));//实例化一个日期

        int day = ca.get(Calendar.DAY_OF_YEAR);//获取是第多少天

        return day;
    }

    /**
     * 获取前一个月的第一天
     *
     * @return
     */
    public static String getMouthFirstday() {

        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        //获取前月的第一天
        Calendar cal_1 = Calendar.getInstance();//获取当前日期
        cal_1.add(Calendar.MONTH, -1);
        cal_1.set(Calendar.DAY_OF_MONTH, 1);//设置为1号,当前日期既为本月第一天
        String firstDay = format.format(cal_1.getTime());
        return firstDay;
    }

    /**
     * 获取前一个月的追后一天
     *
     * @return
     */
    public static String getMouthLastday() {

        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        //获取前月的最后一天
        Calendar cale = Calendar.getInstance();//获取当前日期
        cale.set(Calendar.DAY_OF_MONTH, 0);//设置为1号,当前日期既为本月第一天
        String lastDay = format.format(cale.getTime());
        return lastDay;
    }

    /**
     * 获取日期所在月的第一天
     *
     * @param date
     * @return
     */
    public static Date getMonthFirstday(Date date) {
        return DateUtils.parse(DateUtils.format(date, DateUtils.P_YYYY_MM) + "-01", DateUtils.P_YYYY_MM_DD);
    }

    public static String getMonthFirstdayToStr(Date date) {
        return DateUtils.format(DateUtils.getMonthFirstday(date), DateUtils.P_YYYY_MM_DD);
    }

    /**
     * 获取日期所在月的最后一天
     *
     * @param date
     * @return
     */
    public static Date getMonthLastday(Date date) {
        return DateUtils.addDay(DateUtils.addMonth(getMonthFirstday(date), 1), -1);
    }

    public static String getMonthLastdayToStr(Date date) {
        return DateUtils.format(DateUtils.getMonthLastday(date), DateUtils.P_YYYY_MM_DD);
    }

    /**
     * 获取系统当前年份
     *
     * @return
     */
    public static String getCurrentYear() {
        Calendar date = Calendar.getInstance();

        String year = String.valueOf(date.get(Calendar.YEAR));
        return year;
    }

    /**
     * 计算两个日期相差几天
     *
     * @param starttime
     * @param endtime
     * @return
     */
    public static int getDays(Date starttime, Date endtime) {
        long nd = 1000 * 24 * 60 * 60;
        Date startdate = new Date();
        Date enddate = new Date();
        startdate = DateUtils.parse(starttime, DateUtils.P_YYYY_MM_DD);
        enddate = DateUtils.parse(endtime, DateUtils.P_YYYY_MM_DD);
        long diff = enddate.getTime() - startdate.getTime();
        return (int) (diff / nd);
    }

    /**
     * 计算两个日期相差几天
     *
     * @param starttime
     * @param endtime
     * @return
     */
    public static String getDays(String starttime, String endtime) {
        long nd = 1000 * 24 * 60 * 60;
        Date startdate = new Date();
        Date enddate = new Date();
        startdate = DateUtils.parse(starttime, DateUtils.P_YYYY_MM_DD);
        enddate = DateUtils.parse(endtime, DateUtils.P_YYYY_MM_DD);
        long diff = enddate.getTime() - startdate.getTime();
        long day = diff / nd;
        String daystr = String.valueOf(day);
        return daystr;
    }

    public static int compareDate(Date date1, Date date2) {
        String str1 = format(date1);
        String str2 = format(date2);
        return str1.compareTo(str2);
    }

    /**
     * 比较两个日期大小
     *
     * @return
     */
    public static int compareDate(String date1, String date2) {
        DateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            Date date3 = sdf.parse(date1);
            Date date4 = sdf.parse(date2);
            // 判断是否有默认时间 1970-01-01
//	       String date5 = DateUtils.format(date3,DateUtils.P_YYYY);
//	       String date6 = DateUtils.format(date4,DateUtils.P_YYYY);
//	       if ("1970".equals(date5)) {
//           return 1;
//	       } else if ("1970".equals(date6)) {
//           return -1;
//	       }
            if (date3.getTime() > date4.getTime()) {
                return 1;
            } else if (date3.getTime() < date4.getTime()) {
                return -1;
            } else {
                return 0;
            }
        } catch (Exception exception) {
            exception.printStackTrace();
        }
        return 0;
    }

    /**
     * 获取上(下)周周几的日期
     *
     * @param dayOfWeek  1-7 2-7代表周一至周六，1代表周日
     * @param weekOffset 周偏移，上周为-1，本周为0，下周为1，以此类推
     * @return
     */
    @SuppressWarnings("unused")
    public static String getDayOfWeek(int dayOfWeek, int weekOffset) {
        if (dayOfWeek > Calendar.SATURDAY || dayOfWeek < Calendar.SUNDAY) {
            return null;
        }
        if (Calendar.MONDAY > Calendar.SATURDAY || Calendar.MONDAY < Calendar.SUNDAY) {
            return null;
        }
        Calendar date = Calendar.getInstance(Locale.CHINA);
        date.setFirstDayOfWeek(Calendar.MONDAY);
        // 周数减一，即上周
        date.add(Calendar.WEEK_OF_MONTH, weekOffset);
        // 日设为周几
        date.set(Calendar.DAY_OF_WEEK, dayOfWeek);
        // 时分秒全部置0
        date.set(Calendar.HOUR_OF_DAY, 0);
        date.set(Calendar.MINUTE, 0);
        date.set(Calendar.SECOND, 0);
        date.set(Calendar.MILLISECOND, 0);
        return DateUtils.format(date.getTime(), DateUtils.P_YYYY_MM_DD);
    }

    /**
     * 是否同一天
     *
     * @param date1
     * @param date2
     * @return
     */
    public static boolean isSameDay(Date date1, Date date2) {
        return format(date1).equals(format(date2));
    }

    /**
     * 判断两个日期是否在同一周期
     *
     * @param date1
     * @param date2
     * @return
     */
    public static boolean isSameDate(String date1, String date2, String pattern) {
        SimpleDateFormat format = new SimpleDateFormat(pattern);
        Date d1 = null;
        Date d2 = null;
        try {
            d1 = format.parse(date1);
            d2 = format.parse(date2);
        } catch (Exception e) {
            e.printStackTrace();
        }
        Calendar cal1 = Calendar.getInstance();
        Calendar cal2 = Calendar.getInstance();
        cal1.setFirstDayOfWeek(Calendar.MONDAY); // 西方周日为一周的第一天，将周一设为一周第一天
        cal2.setFirstDayOfWeek(Calendar.MONDAY);
        cal1.setTime(d1);
        cal2.setTime(d2);
        int subYear = cal1.get(Calendar.YEAR) - cal2.get(Calendar.YEAR);
        if (subYear == 0) { // subYear==0,说明是同一年
            if (cal1.get(Calendar.WEEK_OF_YEAR) == cal2.get(Calendar.WEEK_OF_YEAR))
                return true;
        } else if (subYear == 1 && cal2.get(Calendar.MONTH) == 11) { // subYear==1,说明cal比cal2大一年;java的一月用"0"标识，那么12月用"11"
            if (cal1.get(Calendar.WEEK_OF_YEAR) == cal2.get(Calendar.WEEK_OF_YEAR))
                return true;
        } else if (subYear == -1 && cal1.get(Calendar.MONTH) == 11) { // subYear==-1,说明cal比cal2小一年
            if (cal1.get(Calendar.WEEK_OF_YEAR) == cal2.get(Calendar.WEEK_OF_YEAR))
                return true;
        }
        return false;
    }

    /**
     * 加小时
     *
     * @param date
     * @param ii
     * @return
     */
    public static Date addHour(Date date, int ii) {
        Calendar ins = Calendar.getInstance();
        ins.setTime(date);
        ins.set(Calendar.HOUR, ins.get(Calendar.HOUR) + ii);
        return ins.getTime();
    }

    /**
     * 将日期转化为上中下旬
     *
     * @return
     * @author wang
     * @date 2021-3-11
     */
    public static String getXun(Date date) {
        String returnCode = "月下旬";

        Calendar cal = Calendar.getInstance();//日历对象
        cal.setTime(date);
        int day = cal.get(Calendar.DATE);//获取日
        if (day < 11)
            returnCode = "月上旬";
        else if (day < 21)
            returnCode = "月中旬";
        returnCode = cal.get(Calendar.YEAR) + "年" + String.valueOf(cal.get(Calendar.MONTH) + 1) + returnCode;

        return returnCode;
    }

    /**
     * 将日期转化为上中下旬
     *
     * @return
     * @author wang
     * @date 2021-3-11
     */
    public static String getXun(String indate) {
        String returnCode = "月下旬";

        Calendar cal = Calendar.getInstance();//日历对象
        try {
            cal.setTime(new SimpleDateFormat("yyyy-MM-dd").parse(indate));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        int day = cal.get(Calendar.DATE);//获取日
        if (day < 11)
            returnCode = "月上旬";
        else if (day < 21)
            returnCode = "月中旬";
        returnCode = cal.get(Calendar.YEAR) + "年" + String.valueOf(cal.get(Calendar.MONTH) + 1) + returnCode;

        return returnCode;
    }

    /**
     * 判断是平年还是闰年
     * @param year
     * @return
     *//*
	public static boolean isLeapYear(String year) {
		boolean flag = false;
		if (!StringUtils.isEmpty(year)) {
			int years = Integer.parseInt(year);
			if ((years % 4 == 0 && years % 100 != 0) || (years % 400 == 0)) {
				flag = true;
			}
		}
		return flag;
	}*/

    /**
     * 根据给定日期获取当前周一
     *
     * @param date
     * @return
     */
    public static String getFirstDayOfWeek(String date) {
        Calendar c = Calendar.getInstance();
        c.setTime(parse(date, P_YYYY_MM_DD));
        if (c.get(Calendar.DAY_OF_WEEK) == 1) {
            c.add(Calendar.DAY_OF_MONTH, -1);
        }
        c.add(Calendar.DATE, c.getFirstDayOfWeek() - c.get(Calendar.DAY_OF_WEEK) + 1);
        return format(c.getTime(), P_YYYY_MM_DD);
    }

    /**
     * 根据给定日期获取当前周日
     *
     * @param date
     * @return
     */
    public static String getLastDayOfWeek(String date) {
        Calendar c = Calendar.getInstance();
        c.setTime(parse(date, P_YYYY_MM_DD));
        // 如果是周日直接返回
        if (c.get(Calendar.DAY_OF_WEEK) == 1) {
            return date;
        }
        c.add(Calendar.DATE, 7 - c.get(Calendar.DAY_OF_WEEK) + 1);
        return format(c.getTime(), P_YYYY_MM_DD);
    }

    public static Date getBeginDate(Date date) {
        return DateUtils.parse(DateUtils.format(date) + " 00:00:00", DateUtils.P_YYYY_MM_DD_HH_MM_SS);
    }

    public static Date GetEndDate(Date date) {
        return DateUtils.parse(DateUtils.format(date) + " 23:59:59", DateUtils.P_YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 将yyyy-MM-dd HH:mm转换为yyyyMMddHHmm
     *
     * @param time
     * @return
     * @throws ParseException
     */
    public static String dateToNum(String time) throws ParseException {
        SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd HH:mm");
        Date date = sdf1.parse(time);
        SimpleDateFormat sdf2 = new SimpleDateFormat("yyyyMMddHHmm");
        String dateStr = sdf2.format(date);
        return dateStr;
    }

    /**
     * 将yyyy/MM/dd HH:mm转换为yyyyMMddHHmm
     *
     * @param time
     * @return
     * @throws ParseException
     */
    public static String dateToNums(String time) throws ParseException {
        SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy/MM/dd HH:mm");
        Date date = sdf1.parse(time);
        SimpleDateFormat sdf2 = new SimpleDateFormat("yyyyMMddHHmm");
        String dateStr = sdf2.format(date);
        return dateStr;
    }

    /**
     * 获取字符串中的数字
     *
     * @param str
     * @return
     */
    public static List<String> getStr(String str) {
        String regEx = "[\u4E00-\u9FA5]";
        Pattern p = Pattern.compile(regEx);
        Matcher m = p.matcher(str);
        str = m.replaceAll("").trim();
        String[] ss = str.split("  ");
        // 把数组放进list里面，进行遍历，把数组里的每个元素都单独取出来
        List<String> strList = new ArrayList<>();
        for (int i = 0; i < ss.length; i++) {
            strList.add(ss[i]);
        }
        return strList;
    }

    /**
     * 获取当前日期的前一天
     *
     * @return
     */
    public static String getBeforeDate() {
        String beforeDate = "";
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        Date date = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DAY_OF_MONTH, -1);
        date = calendar.getTime();
        beforeDate = sdf.format(date);
        return beforeDate;
    }

    /**
     * 将yyyyMMdd转换为yyyy-MM-dd
     *
     * @param time
     * @return
     * @throws ParseException
     */
    public static String numToDate(String time) throws ParseException {
        SimpleDateFormat sdf1 = new SimpleDateFormat("yyyyMMdd");
        Date date = sdf1.parse(time);
        SimpleDateFormat sdf2 = new SimpleDateFormat("yyyy-MM-dd");
        String dateStr = sdf2.format(date);
        return dateStr;
    }

    /**
     * 将yyyy-MM-dd字符串转换为日期
     *
     * @param dateStr
     * @return
     * @throws ParseException
     */
    public static Date parseDate(String dateStr) {
        return parse(dateStr, P_YYYY_MM_DD);
    }

    public static String getNearDate(List<String> dateList, String date, String pattern) {

        long minDifference = Long.MAX_VALUE;
        Date nearestDateTime = null;
        Date givenDateTime = DateUtils.parse(date, pattern);
        for (String time : dateList) {
            Date dateTime = DateUtils.parse(time, pattern);
            ;

            int difference = Math.abs(DateUtils.getDays(givenDateTime, dateTime));
            if (difference < minDifference) {
                minDifference = difference;
                nearestDateTime = dateTime;
            }
        }
        return DateUtils.format(nearestDateTime, pattern);
    }

    /**
     * 根据时间获取指定日期的周一
     *
     * @return
     * @throws Exception
     */
    public static Date getMondayByYMD(Date ymd) throws ParseException {
        Calendar c = Calendar.getInstance();
        c.setTime(ymd);
        if (c.get(Calendar.DAY_OF_WEEK) == 1) {
            c.add(Calendar.DAY_OF_MONTH, -1);
        }
        c.add(Calendar.DATE, c.getFirstDayOfWeek() - c.get(Calendar.DAY_OF_WEEK) + 1);
        return c.getTime();
    }

    /**
     * 将yyyy/MM/dd HH:mm:ss转换为yyy-MM-dd HH:mm:ss
     *
     * @return
     * @throws ParseException
     */
    public static Date parse(String dateStr) throws ParseException {
        SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
        Date date = sdf1.parse(dateStr);
        SimpleDateFormat sdf2 = new SimpleDateFormat(DateUtils.P_YYYY_MM_DD_HH_MM_SS);
        return parse(sdf2.format(date), DateUtils.P_YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 将中文日期yyyy-mm-dd转为中文：二〇二〇年四月二十日
     *
     * @return
     */
    public static String parseDateToZw() {
        Map<String, String> map = new HashMap<>();
        map.put("0", "〇");
        map.put("1", "一");
        map.put("2", "二");
        map.put("3", "三");
        map.put("4", "四");
        map.put("5", "五");
        map.put("6", "六");
        map.put("7", "七");
        map.put("8", "八");
        map.put("9", "九");
        map.put("10", "十");
        map.put("11", "十一");
        map.put("12", "十二");
        map.put("13", "十三");
        map.put("14", "十四");
        map.put("15", "十五");
        map.put("16", "十六");
        map.put("17", "十七");
        map.put("18", "十八");
        map.put("19", "十九");
        map.put("20", "二十");
        map.put("21", "二十一");
        map.put("22", "二十二");
        map.put("23", "二十三");
        map.put("24", "二十四");
        map.put("25", "二十五");
        map.put("26", "二十六");
        map.put("27", "二十七");
        map.put("28", "二十八");
        map.put("29", "二十九");
        map.put("30", "三十");
        map.put("31", "三十一");
        Calendar calendar = Calendar.getInstance();
        String year = calendar.get(Calendar.YEAR) + "";
        String month = (calendar.get(Calendar.MONTH) + 1) + "";
        String day = calendar.get(Calendar.DATE) + "";

        String[] years = year.split("");

        String datestr = "";
        for (int i = 0; i < years.length; i++) {
            datestr = datestr + map.get(years[i]);
        }
        datestr = datestr + "年";
        datestr = datestr + map.get(month);
        datestr = datestr + "月";
        datestr = datestr + map.get(day);
        datestr = datestr + "日";
        return datestr;
    }


    /**
     * 当月第一天
     *
     * @return
     */
    public static String getFirstDay() {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        Calendar calendar = Calendar.getInstance();
        Date theDate = calendar.getTime();

        GregorianCalendar gcLast = (GregorianCalendar) Calendar.getInstance();
        gcLast.setTime(theDate);
        gcLast.set(Calendar.DAY_OF_MONTH, 1);
        String day_first = df.format(gcLast.getTime());
        StringBuffer str = new StringBuffer().append(day_first);
        return str.toString();

    }

    /**
     * 当月第一天
     *
     * @return
     */
    public static String getFirstDay(String yearMonth) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, Integer.parseInt(yearMonth.substring(0, 4)));
        calendar.set(Calendar.MONTH, Integer.parseInt(yearMonth.substring(4, 6)) - 1);
        Date theDate = calendar.getTime();

        GregorianCalendar gcLast = (GregorianCalendar) Calendar.getInstance();
        gcLast.setTime(theDate);
        gcLast.set(Calendar.DAY_OF_MONTH, 1);
        String day_first = df.format(gcLast.getTime());
        StringBuffer str = new StringBuffer().append(day_first);
        return str.toString();

    }

    /**
     * 当月最后一天
     *
     * @return
     */
    public static String getLastDay() {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        Calendar calendar = Calendar.getInstance();
        //Date theDate = calendar.getTime();
        //String s = df.format(theDate);
        calendar.add(Calendar.MONTH, 1);    //加一个月
        calendar.set(Calendar.DATE, 1);        //设置为该月第一天
        calendar.add(Calendar.DATE, -1);    //再减一天即为上个月最后一天
        String day_last = df.format(calendar.getTime());
        StringBuffer endStr = new StringBuffer().append(day_last);
        day_last = endStr.toString();
        StringBuffer str = new StringBuffer().append(day_last);
        return str.toString();

    }

    /**
     * 当月最后一天
     *
     * @return
     */
    public static String getLastDay(String yearMonth) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, Integer.parseInt(yearMonth.substring(0, 4)));
        calendar.set(Calendar.MONTH, Integer.parseInt(yearMonth.substring(4, 6)));
//       calendar.add(Calendar.MONTH, 1);    //加一个月
        calendar.set(Calendar.DATE, 1);        //设置为该月第一天
        calendar.add(Calendar.DATE, -1);    //再减一天即为上个月最后一天
        String day_last = df.format(calendar.getTime());
        StringBuffer endStr = new StringBuffer().append(day_last);
        day_last = endStr.toString();
        StringBuffer str = new StringBuffer().append(day_last);
        return str.toString();

    }

    /**
     * 获取当前年
     *
     * @return
     */
    public static String getYear() {
        Calendar cal = Calendar.getInstance();
        int year = cal.get(Calendar.YEAR);
        return String.valueOf(year);
    }

    /**
     * 获取年的第一天
     *
     * @return
     */
    public static String getDayBeginYear(String day) {
        Date parse = DateUtils.parse(day, P_YYYY_MM_DD);
        String format = DateUtils.format(parse, P_YYYY);
        return format + "-01-01";
    }

    /**
     * 获取当前月
     *
     * @return
     */
    public static String getMonth() {
        Calendar cal = Calendar.getInstance();
        int month = cal.get(Calendar.MONTH);
        return String.valueOf(month + 1);
    }

    /**
     * 获取每月的第一个周三
     *
     * @param month
     * @return
     */
    public static String getMonthWed(int month) {
        Calendar c = Calendar.getInstance();
        c.setTime(new Date());
        c.add(Calendar.MONTH, month);
        c.set(Calendar.DAY_OF_MONTH, 1);
        while (c.get(Calendar.DAY_OF_WEEK) != Calendar.WEDNESDAY) {
            c.add(Calendar.DAY_OF_MONTH, 1);
        }
        return DateUtils.format(c.getTime());
    }

    /**
     * 判断给定日期是否是当月的最后一天
     *
     * @param date
     * @return
     */
    public static boolean isLastDayOfMonth(Date date) {
        //1、创建日历类
        Calendar calendar = Calendar.getInstance();
        //2、设置当前传递的时间，不设就是当前系统日期
        calendar.setTime(date);
        //3、data的日期是N，那么N+1【假设当月是30天，30+1=31，如果当月只有30天，那么最终结果为1，也就是下月的1号】
        calendar.set(Calendar.DATE, (calendar.get(Calendar.DATE) + 1));
        //4、判断是否是当月最后一天【1==1那么就表明当天是当月的最后一天返回true】
        if (calendar.get(Calendar.DAY_OF_MONTH) == 1) {
            return true;
        } else {
            return false;
        }
    }

}
